/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef VNCWIFIDISPLAYDECODER_H_70571462_0029_4B31_B79F_6E6701933255
#define VNCWIFIDISPLAYDECODER_H_70571462_0029_4B31_B79F_6E6701933255

/**
 * \file vncwifidisplaydecoder.h
 *
 * This header file defines types for use with VNC WiFi Display Decoders
 * provided to the VNC WiFi Display SDK.
 *
 * \see vncwifidisplaysdk.h.
 *
 * \section section_lifecycle Lifecycle Diagram
 *
 * A diagram depicting the lifecycle of a VNC WiFi Display Decoder is included
 * below.
 *
 * \dot
 * digraph G {
 *   label = "Decoder Plugin Lifecycle"
 *   color = grey
 *   "Plugin Loaded" -> "Factory Initialized" [label="FactoryInitialize"]
 *   "Factory Initialized" -> "Plugin Loaded" [label="FactoryTerminate"]
 *   "Factory Initialized" -> "Stream Stopped" [label="DecoderCreate"]
 *   "Stream Stopped" -> "Factory Initialized" [label="DecoderDestroy"]
 *   "Stream Stopped" -> "Stream Playing" [label="StreamStart"]
 *   "Stream Playing" -> "Cleanup Needed" [label="StreamPause"]
 *   "Cleanup Needed" -> "Stream Paused" [label="StreamCleanup"]
 *   "Stream Paused" -> "Stream Playing" [label="StreamResume"]
 *   "Stream Paused" -> "Stream Stopped" [label="StreamStop"]
 * }
 * \enddot
 *
 * \if ANDROID
 * \section section_android_viewer_context Android Decoder Viewer Context
 *
 * On Android platforms, your decoder shall receive a
 * VNCWiFiDisplayDecoderAndroidViewerContext instance as the decoder viewer
 * context in its implementation of VNCWiFiDisplayDecoderFactoryInitialize().
 * Your decoder can use the JavaVM instance contained within this context to
 * communicate with your application via the Java Native Interface (JNI). For
 * example, your application may set vendor-specific decoder properties whose
 * values are Java objects. Your decoder can then use JNI to interact directly
 * with these objects.
 *
 * \subsection subsection_native_threads Native Threads
 *
 * Each native thread that performs JNI operations must be attached to the Java
 * Virtual Machine (JVM). The SDK thread shall always be attached to the JVM,
 * and your decoder must never attempt to detach this thread. Your decoder must
 * attach and detach all other native threads (including the RTP client thread)
 * on which JNI operations are performed. If you wish to use JNI in decoder
 * callbacks invoked from the RTP client thread, then you should attach the
 * thread to the JVM at the start of the callback and detach the thread again
 * before returning from the callback. 
 *
 * \see VNCWiFiDisplayDecoderAndroidViewerContext
 * \see VNCWiFiDisplayDecoderFactoryInitializeType
 * \endif
 */

#include "vncwifidisplaytypes.h"
#include "vncwifidisplayrtp.h"

#ifdef __cplusplus
extern "C"
{
#endif

 /**
 * \typedef VNCWiFiDisplayDecoderEventHandle
 * \brief Event handle used by VNC WiFi Display Decoders.
 *
 * A VNCWiFiDisplayDecoderEventHandle is a platform-specific mechanism for
 * the SDK to detect that there has been activity on a WiFi Display Decoder's
 * context.
 *
 * See VNCWiFiDisplayDecoderGetEventHandles() for a full discussion of how
 * the SDK uses VNCWiFiDisplayDecoderEventHandle.
 *
 * \see VNCWiFiDisplayDecoderGetEventHandles(), VNCWiFiDisplayDecoderActivity()
 */
#ifdef _WIN32
    #include <windows.h>
    typedef HANDLE VNCWiFiDisplayDecoderEventHandle;
#else
    typedef int VNCWiFiDisplayDecoderEventHandle;
#endif

/**
 * \brief An enumeration used to describe the direction of activity
 * of interest for each VNCWiFiDisplayDecoderEventHandle.
 *
 * Due to VNCWiFiDisplayDecoderEventHandle being waitable kernel objects on
 * Windows, using any value which is not VNCWiFiDisplayDecoderEventDirectionNone
 * will cause the kernel object to be waited on.
 */
typedef enum
{
  /**
   * Used to indicate that the event handle should not be monitored.
   */
  VNCWiFiDisplayDecoderEventDirectionNone  = 0x0,
  /**
   * Used to indicate that the event handle should only be monitored
   * for read activity.
   */
  VNCWiFiDisplayDecoderEventDirectionRead  = 0x1,
  /**
   * Used to indicate that the event should only be monitored for
   * write activity.
   */
  VNCWiFiDisplayDecoderEventDirectionWrite = 0x2,
  /**
   * Used to indicate that the event should
   * be monitored for both read and write activity.
   */
  VNCWiFiDisplayDecoderEventDirectionBoth  = 0x3
} VNCWiFiDisplayDecoderEventDirection;


/**
 * \brief An enumeration denoting a WiFi Display Decoder property.
 *
 * The viewer application can set these properties using
 * VNCWiFiDisplaySinkSetDecoderProperty().
 *
 * The type of value to provide for the pNewValue argument is specified below
 * for each property type.
 *
 * It is possible to define custom properties, with values starting from
 * VNCWiFiDisplayDecoderPropertyVENDOR.
 */
typedef enum
{
  /**
   * Used to set the visibility of the overlay. The value provided should point
   * to a vnc_bool_t value, set to true if the overlay should be made visible,
   * and false if the overlay should be hidden.
   */
  VNCWiFiDisplayDecoderPropertyOverlayVisibility = 1,
  /**
   * Used to set the position and size of the overlay. The value provided
   * should point to a VNCRectangle, which will contain the desired screen
   * coordinates where the overlay should be placed.
   */
  VNCWiFiDisplayDecoderPropertyOverlayRectangle = 2,
  /**
   * Vendors can define custom properties, with keys greater than this value.
   */
  VNCWiFiDisplayDecoderPropertyOverlayVENDOR = 0x10000,
} VNCWiFiDisplayDecoderProperty;


/**
 * \brief Represents a loaded WiFi Display Decoder plugin.
 *
 * This type is opaque to the SDK, which will pass the value returned
 * by VNCWiFiDisplayDecoderFactoryInitialize() to any further decoder
 * interface calls that may require it.
 */
typedef struct VNCWiFiDisplayDecoderFactoryImpl VNCWiFiDisplayDecoderFactory;

/**
 * \brief Represents a context created using a WiFi Display Decoder.
 *
 * This type is opaque to the SDK, which will pass the value returned by
 * VNCWiFiDisplayDecoderCreate() to any further decoder interface calls
 * that may require it.
 */
typedef struct VNCWiFiDisplayDecoderImpl VNCWiFiDisplayDecoder;

/**
 * \brief Identifies the internal data structures of the SDK that correspond to
 * a loaded WiFi Display Decoder factory.
 *
 * This type is opaque to the decoder implementation, which must pass
 * the value supplied to VNCWiFiDisplayDecoderFactoryInitialize() to any
 * supporting API calls that may require it.
 */
typedef struct VNCWiFiDisplayDecoderFactoryContextImpl *VNCWiFiDisplayDecoderFactoryContext;

/**
 * \brief Identifies the internal data structures of the SDK that correspond to
 * a loaded WiFi Display Decoder.
 *
 * This type is opaque to the decoder implementation, which must pass the value
 * supplied to VNCWiFiDisplayDecoderCreate() to any supporting API calls that
 * may require it.
 */
typedef struct VNCWiFiDisplayDecoderContextImpl *VNCWiFiDisplayDecoderContext;

/**
 * \brief Identifies the WiFi Display Decoder viewer context provided by the
 * user of the WiFi Display SDK.
 *
 * This type is opaque to the SDK, which will just pass the pointer provided to
 * the call to VNCWiFiDisplaySinkRegisterDecoder() or
 * VNCWiFiDisplaySinkLoadDecoderFromFile().
 */
typedef struct VNCWiFiDisplayDecoderViewerContextImpl *VNCWiFiDisplayDecoderViewerContext;

#ifdef ANDROID

#include <jni.h>

/**
 * \brief The viewer context provided to WiFi Display Decoder implementations
 * on Android platforms.
 *
 * On Android platforms, your decoder shall receive an instance of this
 * structure in its implementation of VNCWiFiDisplayDecoderFactoryInitialize().
 * This shall remain valid for the lifetime of your decoder factory.
 *
 * See \ref section_android_viewer_context for more information.
 *
 * \see VNCWiFiDisplayDecoderFactoryInitializeType
 */
typedef struct
{
  /**
   * \brief The size of this structure in bytes.
   */
  const size_t contextSize;

  /**
   * \brief The JavaVM instance to use for JNI operations.
   *
   * On Android platforms, your decoder can use JNI to communicate with your
   * application.
   */
  JavaVM *const pJavaVM;

} VNCWiFiDisplayDecoderAndroidViewerContext;

#endif

/**
 * \brief Error codes that may be returned by the implementations of functions
 * in VNCWiFiDisplayDecoderInterface and VNCWiFiDisplayDecoderSupportingAPI.
 *
 * The error codes are split into two ranges:
 *
 *  - 0 to (VNCWiFiDisplayDecoderErrorVENDOR - 1) - these error codes are a mixture of
 *    error codes (e.g.  VNCWiFiDisplayDecoderErrorNoMemory) and
 *    other error conditions that are pre-defined by RealVNC-provided decoders.
 *    You are encouraged to reuse error codes in this range, rather than
 *    defining your own, as long as it makes sense to do so.
 *  - VNCWiFiDisplayDecoderErrorVENDOR and above - this range of error codes is reserved
 *    for the use of third parties developing decoders.  It is intended for
 *    error conditions that are specific to particular decoder implementations,
 *    and that do not map closely to the codes in the 0 to
 *    (VNCWiFiDisplayDecoderErrorVENDOR - 1) range.
 */
typedef enum
{
    /**
     * \brief The operation was successful - no error occurred.
     */
    VNCWiFiDisplayDecoderErrorNone = 0,

    /**
     * \brief No memory was available for the requested operation.
     */
    VNCWiFiDisplayDecoderErrorNoMemory = 1,

    /**
     * \brief The required resources are unavailable.
     */
    VNCWiFiDisplayDecoderErrorResourceUnavailable = 2,

    /**
     * \brief The decoder requires a licensed feature for which the SDK does
     * not have a license.
     */
    VNCWiFiDisplayDecoderErrorFeatureNotLicensed = 3,

    /**
     * \brief The value of one of the parameters to the call is not valid.
     */
    VNCWiFiDisplayDecoderErrorInvalidParameter = 4,

    /**
     * \brief The SDK encountered an unknown internal error.
     */
    VNCWiFiDisplayDecoderErrorSDKInternal = 5,

    /**
     * \brief There was a problem with the incoming data.
     */
    VNCWiFiDisplayDecoderErrorMalformedData = 6,

    /**
     * \brief The required resources encountered an error.
     */
    VNCWiFiDisplayDecoderErrorResourceError = 7,

    /**
     * \brief The decoder encountered an unknown internal error.
     */
    VNCWiFiDisplayDecoderErrorInternal = 8,

    /**
     * \brief The action cannot be performed in the current state.
     */
    VNCWiFiDisplayDecoderErrorIllegalState = 9,

    /**
     * \brief The codec could not be added as another codec in the list has the
     * same format.
     */
    VNCWiFiDisplayDecoderErrorCodecAlreadyExists = 10,

    /**
     * \brief The codec could not be added as it is missing support for a
     * mandatory mode.
     */
    VNCWiFiDisplayDecoderErrorCodecMissingMandatoryMode = 11,

    /**
     * \brief Start of range of third-party decoder-specific error codes.
     */
    VNCWiFiDisplayDecoderErrorVENDOR = 0x10000
} VNCWiFiDisplayDecoderError;

/**
 * \brief A structure describing the WiFi Display Decoder.
 */
typedef struct
{
  /**
   * The human readable name for this implementation of the decoder,
   * encoded as modified UTF-8 with a NUL terminator.
   *
   * This might include the company name or details of the specific
   * hardware requirements. This will only be used by the SDK in
   * logging and debugging messages.
   *
   * This should have been allocated using VNCWiFiDisplayDecoderFactoryAlloc().
   */
  char *pDecoderName;
  /**
   * The human readable version for this implementation of the decoder,
   * encoded as modified UTF-8 with a NUL terminator.
   *
   * The format of this string is entirely up to the decoder, it will
   * only be used by the SDK in logging and debugging messages.
   *
   * This should have been allocated using VNCWiFiDisplayDecoderFactoryAlloc().
   */
  char *pDecoderVersion;
  /**
   * If this is true, the SDK will not do any internal buffering or re-ordering
   * of RTP packets. If this is false, the SDK will wait a short time for
   * packets which are missing or out-of-order.
   *
   * It is highly recommended that this is set to false.
   */
  vnc_bool_t disableInternalRTPBuffering;

} VNCWiFiDisplayDecoderSupportInfo;


/**
 * \brief Called by the SDK prior to unloading a decoder DLL or shared object.
 *
 * The implementation should free all resources associated with the
 * decoder implementation.
 *
 * The SDK will call VNCWiFiDisplayDecoderDestroy() for all
 * VNCWiFiDisplayDecoder objects created by this decoder before calling this
 * method.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoderFactory The decoder factory to be unloaded.
 */
typedef void VNCCALL
VNCWiFiDisplayDecoderFactoryTerminate(
    VNCWiFiDisplayDecoderFactory *pDecoderFactory);

/**
 * \brief Called by the SDK to request that the decoder factory fills in
 * a structure with support information.
 *
 * Ownership of any strings contained in pDecoderSupportInfo is transferred
 * to the SDK if VNCWiFiDisplayDecoderErrorNone is returned.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoderFactory The decoder factory to request information about.
 * \param pDecoderSupportInfo On success this should have been filled
 * with information describing the decoder.
 * \param decoderSupportInfoSize The size of the memory pointed to by
 * pDecoderSupportInfo.
 *
 * \return VNCWiFiDisplayDecoderErrorNone on success, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderFactoryGetSupportInfo(
    VNCWiFiDisplayDecoderFactory *pDecoderFactory,
    VNCWiFiDisplayDecoderSupportInfo *pDecoderSupportInfo,
    size_t decoderSupportInfoSize);

/**
 * \brief Called by the SDK at some point during or after the start of
 * a connection to create a per-connection instance of a decoder.
 *
 * Failure to create a decoder will be treated by the VNC WiFi Display Sink as
 * a fatal error.
 *
 * The created decoder is initially in the "stream stopped" state. See the
 * \ref section_lifecycle for more details.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoderFactory The decoder factory to use.
 * \param decoderContext Opaque data type to be provided to applicable
 * supporting APIs.
 *
 * \return A pointer for use in other decoder calls by the SDK, or
 * NULL on failure.
 *
 * \see VNCWiFiDisplayDecoderDestroy
 */
typedef VNCWiFiDisplayDecoder *VNCCALL
VNCWiFiDisplayDecoderCreate(
    VNCWiFiDisplayDecoderFactory *pDecoderFactory,
    VNCWiFiDisplayDecoderContext decoderContext);

/**
 * \brief Called by the SDK to destroy a previously created decoder.
 *
 * This will be called by the SDK as part of a connection being
 * destroyed. The pointer provided will match one returned from a
 * previous call to VNCWiFiDisplayDecoderCreate().
 *
 * This callback will only be invoked while the stream is in the stopped state.
 * See the \ref section_lifecycle for more details.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder The decoder to destroy.
 */
typedef void VNCCALL
VNCWiFiDisplayDecoderDestroy(VNCWiFiDisplayDecoder *pDecoder);

/**
 * \brief Called by the SDK to retrieve the current
 * VNCWiFiDisplayDecoderEventHandle objects for the given decoder.
 *
 * Each decoder can provide zero or more handles for the SDK to
 * monitor for it. The SDK obtains these event handles by calling this
 * method on every iteration of its main loop. When activity is detected
 * on these handles the SDK will call VNCWiFiDisplayDecoderActivity(). This
 * prevents the decoder from needing to run a separate thread for performing
 * such asynchronous actions.
 *
 * On Windows, VNCWiFiDisplayDecoderEventHandle is a HANDLE that must identify
 * a waitable kernel object. On UNIX platforms,
 * VNCWiFiDisplayDecoderEventHandle is a file descriptor suitable for use in a
 * select() call.
 *
 * On successful return, *ppEventHandles should be set to point to an
 * array of handles and *ppEventDirections should point to an equal
 * sized array of VNCWiFiDisplayDecoderEventDirection values. The decoder
 * retains ownership of the arrays, and they should remain valid until the next
 * call to VNCWiFiDisplayDecoderGetEventHandles() or
 * VNCWiFiDisplayDecoderDestroy() for the decoder.
 *
 * Returning an error from this callback will cause the WFD sink to terminate.
 * Other callbacks may fire as a result of the shutdown process.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder The decoder to use.
 * \param ppEventHandles An out parameter pointing to an array of event handles.
 * \param ppEventDirection An out parameter pointing to an array indicating the
 * direction of interest for the handles in ppEventHandles.
 * \param eventHandleCount The number of event handles in the arrays pointed to
 * by *ppEventHandles and *ppEventDirection.
 * \return VNCWiFiDisplayDecoderErrorNone on success, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderGetEventHandles(
    VNCWiFiDisplayDecoder *pDecoder,
    const VNCWiFiDisplayDecoderEventHandle **ppEventHandles,
    const VNCWiFiDisplayDecoderEventDirection **ppEventDirections,
    size_t *eventHandleCount);

/**
 * \brief Called by the SDK when activity is detected on an event handle.
 *
 * When activity is detected in the appropriate direction on one or
 * more of the event handles provided by the previous call to
 * VNCWiFiDisplayDecoderGetEventHandles() then the SDK will call this method.
 *
 * The decoder should take care to clear any pending activity on the
 * handles or to not include them in the next call to
 * VNCWiFiDisplayDecoderGetEventHandles(), otherwise the SDK will call this
 * method almost immediately again.
 *
 * Returning an error from this callback will cause the WFD sink to terminate.
 * Other callbacks may fire as a result of the shutdown process.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder The decoder to use.
 * \return VNCWiFiDisplayDecoderErrorNone on success, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderActivity(VNCWiFiDisplayDecoder *pDecoder);

/**
 * \brief Called by the SDK when the viewer application sets a decoder property,
 * using VNCWiFiDisplaySinkSetDecoderProperty().
 *
 * Any operations needed to accommodate the new value should be performed
 * in response to receiving this method call.
 *
 * \if ANDROID
 * On Android platforms, if the property is vendor-specific then newValue shall
 * be a jobject* (a pointer to a JNI reference to a Java object), and valueSize
 * shall be equal to sizeof(jobject). Your decoder can use JNI to interact with
 * this Java object.
 *
 * The JNI reference to the Java object shall be invalidated after this
 * callback returns. If you wish to use the object in native code outside of
 * this callback, then you must create a new global reference to it.
 *
 * See \ref section_android_viewer_context for more information on using JNI in
 * your decoder implementation.
 * \endif
 *
 * Returning an error from this callback will NOT result in the sink
 * terminating.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder The decoder to use.
 * \param propertyKey The key representing the property being set.
 * \param newValue A pointer to the new value of the property. This is owned
 * by the viewer application, and will be valid for the duration of this
 * method call.
 * \param valueSize The size of the value pointed to by newValue, in bytes.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderPropertySet(VNCWiFiDisplayDecoder *pDecoder,
                                 VNCWiFiDisplayDecoderProperty propertyKey,
                                 const void* newValue,
                                 size_t valueSize);

/**
 * \brief Called by the SDK to inform the decoder that the stream has started
 * playing.
 *
 * While the stream is in the playing state, incoming RTP packets should be
 * decoded and rendered.
 *
 * See the decoder \ref section_lifecycle for more information.
 *
 * Returning an error from this callback will cause the WFD sink to terminate.
 * Other callbacks may fire as a result of the shutdown process.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder The decoder to use.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderStreamStart(VNCWiFiDisplayDecoder *pDecoder);

/**
 * \brief Called by the SDK to inform the decoder that the stream has been
 * paused.
 *
 * This may be in response to an RTSP pause event, or may be for some other
 * reason (for instance, if the RTP client detects a significant discontinuity
 * in RTP timestamps or sequence numbers, it may choose to restart the stream).
 *
 * An invocation of this callback will always be followed by an invocation of
 * VNCWiFiDisplayDecoderStreamCleanup() (from the RTP client thread) before any
 * other callbacks are invoked.
 *
 * While the stream is in the paused state, incoming RTP packets may be
 * ignored.
 *
 * See the decoder \ref section_lifecycle for more information.
 *
 * Returning an error from this callback will cause the WFD sink to terminate.
 * Other callbacks may fire as a result of the shutdown process.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder the decoder to use.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderStreamPause(VNCWiFiDisplayDecoder *pDecoder);

/**
 * \brief Called by the RTP client to inform the decoder that any RTP packet
 * structures owned by the decoder should be immediately returned to the SDK.
 *
 * This callback is invoked from the RTP thread.
 *
 * This callback will be invoked immediately after an invocation of
 * VNCWiFiDisplayDecoderStreamPause(). Although this callback is invoked from
 * the RTP thread, other callbacks will not occur simultaneously -- both the
 * SDK thread and the RTP thread will block until this callback returns.
 *
 * The only supporting API which should be called from this method is
 * VNCWiFiDisplayDecoderRecycleRTPPacket(). Calling any other supporting APIs
 * will lead to undefined behaviour.
 *
 * All RTP packets owned by the decoder should be returned to the SDK (using
 * VNCWiFiDisplayDecoderRecycleRTPPacket()) by the time this method returns.
 * If any packets are still owned by the decoder at this time, the SDK may
 * treat this as a fatal error to avoid a memory leak.
 *
 * See the decoder \ref section_lifecycle for more information.
 *
 * Returning an error from this callback will cause the WFD sink to terminate.
 * Other callbacks may fire as a result of the shutdown process.
 *
 * \param pDecoder The decoder to use.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderStreamCleanup(VNCWiFiDisplayDecoder *pDecoder);

/**
 * \brief Called by the SDK to inform the decoder that the stream has been
 * un-paused.
 *
 * An invocation of this callback will always occur following an invocation of
 * VNCWiFiDisplayDecoderStreamCleanup(). Therefore, the stream will always be
 * in the paused state when this callback is invoked.
 *
 * After this callback is invoked, the decoder should resume the decoding and
 * rendering of packets.
 *
 * See the decoder \ref section_lifecycle for more information.
 *
 * Returning an error from this callback will cause the WFD sink to terminate.
 * Other callbacks may fire as a result of the shutdown process.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder the decoder to use.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderStreamResume(VNCWiFiDisplayDecoder *pDecoder);

/**
 * \brief Called by the SDK to inform the decoder that the stream has been
 * stopped.
 *
 * This callback will only be invoked while the stream is in the paused state.
 * VNCWiFiDisplayDecoderStreamCleanup() will therefore already have been
 * invoked by the time this is called, and so the decoder should not be in
 * possession of any RTP packets at this time.
 *
 * The stream may later be restarted using VNCWiFiDisplayDecoderStreamStart(),
 * or the decoder may be destroyed using VNCWiFiDisplayDecoderDestroy().
 *
 * See the decoder \ref section_lifecycle for more information.
 *
 * Returning an error from this callback will cause the WFD sink to terminate.
 * Other callbacks may fire as a result of the shutdown process.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder the decoder to use.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderStreamStop(VNCWiFiDisplayDecoder *pDecoder);

/**
 * \brief Called by the RTP client when an RTP packet is received.
 *
 * The WiFi Display Decoder should handle the demuxing, decoding, and rendering
 * of RTP data.
 *
 * Ownership of the RTP packet is transferred to the WiFi Display Decoder, and
 * when it is no longer needed, this should be returned to the SDK using the
 * VNCWiFiDisplayDecoderRecycleRTPPacket() supporting API. Failure to return
 * the packet to the SDK will result in a memory leak. Ownership is transferred
 * whether or not this callback returns an error.
 *
 * The SDK will only provide RTP packets while the stream is in the playing
 * state. RTP packets received while the stream is paused or stopped will be
 * discarded.
 *
 * Buffering and re-ordering of RTP packets is handled internally by the WiFi
 * Display SDK, and so it is expected that the decoder will decode and render
 * the data as soon as possible.
 *
 * This callback is invoked from the RTP client thread, and may therefore
 * be called in parallel/simultaneously with other callbacks, such as
 * VNCWiFiDisplayDecoderActivity(). Care should be taken to avoid race
 * conditions between callbacks, and state shared between multiple threads
 * should be avoided.
 *
 * Only the VNCWiFiDisplayDecoderRecycleRTPPacket() and
 * VNCWiFiDisplayDecoderLog() supporting APIs may be called from this method.
 * Calling other supporting APIs will result in undefined behaviour.
 *
 * This method should return as quickly as possible, since blocking the RTP
 * client thread could result in dropped packets.
 *
 * For more information on the VNCWiFiDisplayRTPPacket struct, see the
 * vncwifidisplayrtp.h header.
 *
 * Returning an error from this callback will cause the WFD sink to terminate
 * shortly afterwards.
 *
 * \param pDecoder The decoder to use.
 * \param pPacket The RTP packet which has been received.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderOnRTPPacket(VNCWiFiDisplayDecoder *pDecoder,
                                 VNCWiFiDisplayRTPPacket *pPacket);

/**
 * \brief Called by the RTP client thread following an invocation of the
 * VNCWiFiDisplayDecoderPostRTPThreadMessage() supporting API.
 *
 * See VNCWiFiDisplayDecoderPostRTPThreadMessage() for more details.
 *
 * This callback is invoked from the RTP client thread. Only the
 * VNCWiFiDisplayDecoderRecycleRTPPacket() and VNCWiFiDisplayDecoderLog()
 * supporting APIs may be called from this method. Calling other supporting
 * APIs will result in undefined behaviour.
 *
 * Additionally, the caller of the VNCWiFiDisplayDecoderPostRTPThreadMessage()
 * will be blocked until this callback returns.
 *
 * As this callback is invoked from the RTP client thread, it may be called at
 * the same time as other callbacks, such as VNCWiFiDisplayDecoderActivity().
 * Care should be taken to avoid race conditions between callbacks, and state
 * shared between multiple threads should be avoided. However, if
 * VNCWiFiDisplayDecoderPostRTPThreadMessage() was called from the SDK thread,
 * then the SDK thread will be blocked and no other callbacks will be received,
 * allowing state changes to be conveyed without race conditions occurring.
 *
 * It is safe to call this method from any SDK-thread callback, including
 * VNCWiFiDisplayDecoderDestroy().
 *
 * This method should return as quickly as possible, since blocking the RTP
 * client thread could result in dropped packets.
 *
 * The error value returned by this callback will be provided to the caller
 * of VNCWiFiDisplayDecoderPostRTPThreadMessage().
 *
 * \param pDecoder The decoder to use.
 * \param messageType The integer passed to
 * VNCWiFiDisplayDecoderPostRTPThreadMessage().
 * \param pData The pointer passed to
 * VNCWiFiDisplayDecoderPostRTPThreadMessage().
 * \param dataSize The size of the data pointed to by pData.
 * \return A value to be passed to the caller of
 * VNCWiFiDisplayDecoderPostRTPThreadMessage().
 */
typedef vnc_uint64_t VNCCALL
VNCWiFiDisplayDecoderHandleRTPThreadMessage(VNCWiFiDisplayDecoder *pDecoder,
                                            vnc_uint32_t messageType,
                                            const void *pData,
                                            vnc_size_t dataSize);

/**
 * \brief Called by the SDK when the WiFi Display source has negotiated, or
 * re-negotiated, the audio codec for the WiFi Display session.
 *
 * The WiFi Display source may select an audio codec that it supports, and
 * which is compatible with at least one of the codecs associated with this VNC
 * WiFi Display Sink by VNCWiFiDisplayDecoderAddAudioCodec(), during the WiFi
 * Display Capability Negotiation phase. When the WiFi Display session is
 * active, the source may also enter a re-negotiation phase to change the
 * selected audio codec. When either of these events occur, the SDK invokes
 * this callback to notify the decoder.
 *
 * The VNCWiFiDisplayAudioCodec instance provided in this callback shall
 * describe the audio codec that has now been selected, and the audio mode(s)
 * that shall be used. The decoder must switch to the appropriate audio
 * codec immediately.
 *
 * If the WiFi Display source does not support audio streaming, then this
 * callback shall not be invoked and audio shall be not be available in the
 * WiFi Display session.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder The decoder to use.
 * \param pCodec The audio codec that has now been selected.
 * \param codecSize The size of codec.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderAudioCodecSelectedCallback(
    VNCWiFiDisplayDecoder *pDecoder,
    const VNCWiFiDisplayAudioCodec *pCodec,
    size_t codecSize);

/**
 * \brief Called by the SDK when the WiFi Display source has negotiated, or
 * re-negotiated, the H.264 codec for the WiFi Display session.
 *
 * The WiFi Display source shall select an H.264 codec that it supports, and
 * which is compatible with at least one of the codecs associated with this VNC
 * WiFi Display Sink by VNCWiFiDisplaySinkAddH264Codec(), during the WiFi
 * Display Capability Negotiation phase. When the WiFi Display session is
 * active, the source may also enter a re-negotiation phase to change the
 * selected H.264 codec. When either of these events occur, the SDK invokes
 * this callback to notify the decoder.
 *
 * The VNCWiFiDisplayH264Codec instance provided in this callback shall
 * describe the H.264 codec that has now been selected, and the properties
 * that shall be used. The decoder must switch to the appropriate H.264
 * codec immediately.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder The decoder to use.
 * \param pCodec The H.264 codec that has now been selected.
 * \param codecSize The size of codec.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderH264CodecSelectedCallback(
    VNCWiFiDisplayDecoder *pDecoder,
    const VNCWiFiDisplayH264Codec *pCodec,
    size_t codecSize);

/**
 * \brief Called by the SDK when the wfd-av-format-change-timing RTSP parameter
 * is used to notify the sink of the time at which an impending AV format
 * change will take effect.
 *
 * Please see the WiFi Display specification (section 6.1.13) for more
 * information.
 *
 * This callback is invoked from the SDK thread.
 *
 * \param pDecoder The decoder to use.
 * \param PTS the most significant 33 bits of the PTS field.
 * \param DTS the most significant 33 bits of the DTS field.
 * \return VNCWiFiDisplayDecoderErrorNone if successful, or a suitable error
 * otherwise.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderAVFormatChangeTimingCallback(
    VNCWiFiDisplayDecoder *pDecoder,
    vnc_uint64_t PTS,
    vnc_uint64_t DTS);

/**
 * \brief APIs to be implemented by the WiFi Display Decoder and called by the
 * SDK or RTP client.
 */
typedef struct {
  /**
   * Called by the SDK prior to unloading a decoder factory DLL or
   * shared object.
   */
  VNCWiFiDisplayDecoderFactoryTerminate *vncWiFiDisplayDecoderFactoryTerminate;
  /**
   * Called by the SDK to retrieve information about the decoder factory
   * support.
   */
  VNCWiFiDisplayDecoderFactoryGetSupportInfo *vncWiFiDisplayDecoderFactoryGetSupportInfo;
  /**
   * Called by the SDK to create a decoder for a connection.
   */
  VNCWiFiDisplayDecoderCreate *vncWiFiDisplayDecoderCreate;
  /**
   * Called by the SDK to destroy a decoder for a connection.
   */
  VNCWiFiDisplayDecoderDestroy *vncWiFiDisplayDecoderDestroy;
  /**
   * Called by the SDK to retrieve a list of VNCWiFiDisplayDecoderEventHandle
   * values for the provided decoder.
   */
  VNCWiFiDisplayDecoderGetEventHandles *vncWiFiDisplayDecoderGetEventHandles;
  /**
   * Called by the SDK to indicate that there is activity on the event handles.
   */
  VNCWiFiDisplayDecoderActivity *vncWiFiDisplayDecoderActivity;
  /**
   * Called by the SDK to indicate that the viewer application has set a
   * decoder property.
   */
  VNCWiFiDisplayDecoderPropertySet *vncWiFiDisplayDecoderPropertySet;
  /**
   * Called by the SDK to indicate that the stream has started.
   */
  VNCWiFiDisplayDecoderStreamStart *vncWiFiDisplayDecoderStreamStart;
  /**
   * Called by the SDK to indicate that the stream has been paused.
   */
  VNCWiFiDisplayDecoderStreamPause *vncWiFiDisplayDecoderStreamPause;
  /**
   * Called by the RTP client thread to indicate that all RTP packet structs
   * should be returned to the SDK immediately.
   */
  VNCWiFiDisplayDecoderStreamCleanup *vncWiFiDisplayDecoderStreamCleanup;
  /**
   * Called by the SDK to indicate that the stream has resumed.
   */
  VNCWiFiDisplayDecoderStreamResume *vncWiFiDisplayDecoderStreamResume;
  /**
   * Called by the SDK to indicate that the stream has stopped.
   */
  VNCWiFiDisplayDecoderStreamStop *vncWiFiDisplayDecoderStreamStop;
  /**
   * Called by the RTP client to indicate that an RTP packet has been received.
   */
  VNCWiFiDisplayDecoderOnRTPPacket *vncWiFiDisplayDecoderOnRTPPacket;
  /**
   * Called by the RTP client thread to pass on a message posted by the
   * VNCWiFiDisplayDecoderPostRTPThreadMessage() method.
   */
  VNCWiFiDisplayDecoderHandleRTPThreadMessage *vncWiFiDisplayDecoderHandleRTPThreadMessage;
  /**
   * Called by the SDK to indicate that the WiFi Display Source has chosen
   * an audio codec.
   */
  VNCWiFiDisplayDecoderAudioCodecSelectedCallback *vncWiFiDisplayDecoderAudioCodecSelectedCallback;
  /**
   * Called by the SDK to indicate that the WiFi Display Source has chosen
   * an H.264 codec.
   */
  VNCWiFiDisplayDecoderH264CodecSelectedCallback *vncWiFiDisplayDecoderH264CodecSelectedCallback;
  /**
   * Called by the SDK to notify the decoder of the time at which an impending
   * AV format change will take effect.
   */
  VNCWiFiDisplayDecoderAVFormatChangeTimingCallback *vncWiFiDisplayDecoderAVFormatChangeTimingCallback;

} VNCWiFiDisplayDecoderInterface;


/**
 * \brief Called by the decoder factory to write to the SDK's log.
 *
 * This supporting API must be called only from the SDK thread.
 *
 * \param decoderFactoryContext The decoder VNCWiFiDisplayDecoderFactoryContext
 * that was passed to VNCWiFiDisplayDecoderFactoryInitialize().
 * \param severity The severity of the log message (see
 * VNCWiFiDisplayParameterLog for an explanation).
 * \param text The text to write to the log, as a NUL-terminated UTF-8
 * string.
 *
 * \see VNCWiFiDisplayParameterLog
 */
typedef void VNCCALL
VNCWiFiDisplayDecoderFactoryLog(
    VNCWiFiDisplayDecoderFactoryContext decoderFactoryContext,
    vnc_int32_t severity,
    const char *text);

/**
 * \brief Called by the decoder to allocate memory whose ownership is to be
 * transferred to the SDK.
 *
 * The start address of the returned memory is aligned to a word boundary, and
 * the memory is filled with zeroes.
 *
 * If ownership has been passed to the SDK, then the SDK will free the allocated
 * memory when it is no longer needed. If ownership is not transferred, then
 * the plugin should free the memory using VNCWiFiDisplayDecoderFactoryFree().
 *
 * This supporting API must be called only from the SDK thread.
 *
 * \param decoderFactoryContext The VNCWiFiDisplayDecoderFactoryContext that
 * was passed to VNCWiFiDisplayDecoderFactoryInitialize().
 * \param size The size of data to be allocated.
 *
 * \return A pointer to the allocated memory or NULL on failure.
 */
typedef void *VNCCALL
VNCWiFiDisplayDecoderFactoryAlloc(
    VNCWiFiDisplayDecoderFactoryContext decoderFactoryContext,
    size_t size);

/**
 * \brief Called by the decoder to allocate memory whose ownership is to be
 * transferred to the SDK.
 *
 * The contents of the allocated memory will be equal to the supplied string,
 * including the NUL terminator.
 *
 * If ownership has been passed to the SDK, then the SDK will free the allocated
 * memory when it is no longer needed. If ownership is not transferred, then
 * the plugin should free the memory using VNCWiFiDisplayDecoderFactoryFree().
 *
 * This supporting API must be called only from the SDK thread.
 *
 * \param decoderFactoryContext The VNCWiFiDisplayDecoderFactoryContext that
 * was passed to VNCWiFiDisplayDecoderFactoryInitialize().
 * \param str The string to copy into the allocated memory.
 *
 * \return A pointer to the allocated memory or NULL on failure.
 */
typedef char *VNCCALL
VNCWiFiDisplayDecoderFactoryAllocString(
    VNCWiFiDisplayDecoderFactoryContext decoderFactoryContext,
    const char* str);

/**
 * \brief Called by the decoder to free memory which was allocated by
 * VNCWiFiDisplayDecoderFactoryAlloc() or
 * VNCWiFiDisplayDecoderFactoryAllocString(), but whose ownership was not
 * transferred to the SDK.
 *
 * RTP packets should NOT be provided to this method. To dispose of RTP
 * packets, use VNCWiFiDisplayDecoderRecycleRTPPacket().
 *
 * This supporting API must be called only from the SDK thread.
 *
 * \param decoderFactoryContext The VNCWiFiDisplayDecoderFactoryContext that
 * was passed to VNCWiFiDisplayDecoderFactoryInitialize().
 * \param buffer The buffer to be freed.
 */
typedef void VNCCALL
VNCWiFiDisplayDecoderFactoryFree(
    VNCWiFiDisplayDecoderFactoryContext decoderFactoryContext,
    void *buffer);

/**
 * \brief Called by the decoder to write to the SDK's log.
 *
 * Decoder implementations are encouraged to log significant events during
 * decoder creation and activation, and any other information that may help
 * offline diagnosis of problems.
 *
 * It is particularly important to log the reason for a failure if
 * the reason cannot easily be conveyed with an error code.
 *
 * VNCWiFiDisplayDecoderLog() is preferred to VNCWiFiDisplayDecoderFactoryLog()
 * because VNCWiFiDisplayDecoderLog() is guaranteed to work correctly if called
 * from any thread. VNCWiFiDisplayDecoderFactoryLog() does not work if called
 * from a thread that was started by the decoder, or from the RTP client thread.
 *
 * \param decoderContext The VNCWiFiDisplayDecoderContext that was passed to
 * VNCWiFiDisplayDecoderCreate().
 * \param severity The severity of the log message (see
 * VNCWiFiDisplayParameterLog for an explanation).
 * \param text The text to write to the log.
 *
 * \see VNCWiFiDisplayParameterLog
 */
typedef void VNCCALL
VNCWiFiDisplayDecoderLog(
    VNCWiFiDisplayDecoderContext decoderContext,
    vnc_int32_t severity,
    const char *text);

/**
 * \brief Called by the decoder to check whether a particular feature is
 * licensed.
 *
 * This method allows you to ensure that your custom decoders are usable only
 * in certain applications.  The feature check succeeds if the SDK is
 * authorized to use at least one of the requested features.  The features are
 * searched for in the licenses available to the SDK.
 *
 * If none of the features are licensed, then the decoder may either continue
 * to operate, restricting functionality as appropriate, or return
 * VNCWiFiDisplayDecoderErrorFeatureNotLicensed to the SDK, in which case the
 * SDK will terminate the session.
 *
 * This supporting API must be called only from the SDK thread.
 *
 * \param decoderContext The VNCWiFiDisplayDecoderContext that was passed to
 * VNCWiFiDisplayDecoderCreate().
 * \param featureID The feature identifier for which to check for a license.
 * \param pResult On successful return, the value pointed to by pResult is
 * non-zero if any of the features is licensed, and zero if not.
 * \retval VNCWiFiDisplayDecoderErrorNone The feature check was made
 * successfully.  This does *not* imply that any of the features are licensed;
 * check the value pointed to by pResult.
 * \retval VNCWiFiDisplayDecoderErrorInvalidParameter featureIDs or pResult is
 * NULL, or featureIDCount is zero.
 *
 * pResult is non-zero if the feature is licensed, or zero if not.
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderLocalFeatureCheck(
    VNCWiFiDisplayDecoderContext decoderContext,
    const unsigned *featureIDs,
    size_t featureIDCount,
    int *pResult);

/**
 * \brief Called by the decoder to return an RTP packet to the SDK.
 *
 * The packet structure and it's data buffer will be recycled to avoid
 * unnecessary allocations.
 *
 * Failure to return unused RTP packets to the SDK will result in a memory
 * leak. Packets should be returned in a timely manner to avoid excess memory
 * consumption.
 *
 * This API must be called only from the RTP client thread. Calling this API
 * from the SDK thread will lead to undefined behaviour.
 *
 * \param decoderContext The VNCWiFiDisplayDecoderContext that was passed to
 * VNCWiFiDisplayDecoderCreate().
 * \param pPacket The RTP packet to recycle. Ownership of the packet is
 * transferred back to the SDK.
 */
typedef void VNCCALL
VNCWiFiDisplayDecoderRecycleRTPPacket(
    VNCWiFiDisplayDecoderContext decoderContext,
    VNCWiFiDisplayRTPPacket *pPacket);

/**
 * \brief Called by the decoder to post a message to the RTP thread.
 *
 * This method allows other threads to communicate with the RTP thread without
 * the need for locking. For instance, it may be used to convey the changes of
 * state in the stream, changes of codec, timers expiring, or resources
 * becoming ready.
 *
 * This method will cause the VNCWiFiDisplayDecoderHandleRTPThreadMessage()
 * callback to be invoked from the RTP client thread. This supporting API will
 * block until VNCWiFiDisplayDecoderHandleRTPThreadMessage() has returned,
 * allowing state changes to be conveyed without race conditions occurring.
 *
 * This method may be called from any thread, including the SDK thread.
 *
 * The integer messageType, the pointer pData, and the associated size
 * dataSize, will be passed unaltered to
 * VNCWiFiDisplayDecoderHandleRTPThreadMessage().
 *
 * \param decoderContext The VNCWiFiDisplayDecoderContext that was passed to
 * VNCWiFiDisplayDecoderCreate().
 * \param messageType An integer to be passed (unaltered) to
 * VNCWiFiDisplayDecoderHandleRTPThreadMessage().
 * \param pData A pointer to be passed (unaltered) to
 * VNCWiFiDisplayDecoderHandleRTPThreadMessage().
 * \param dataSize The size of the data (in bytes) pointed to by pData.
 * \return The value returned by VNCWiFiDisplayDecoderHandleRTPThreadMessage().
 */
typedef vnc_uint64_t VNCCALL
VNCWiFiDisplayDecoderPostRTPThreadMessage(
    VNCWiFiDisplayDecoderContext decoderContext,
    vnc_uint32_t messageType,
    const void *pData,
    vnc_size_t dataSize);

/**
 * \brief Indicates that this decoder supports a particular audio codec.
 *
 * This method should be called only from
 * VNCWiFiDisplayDecoderFactoryInitialize(). It is not possible to subsequently
 * alter the list of supported codecs.
 *
 * The list of supported codecs will be made available to the application using
 * the VNCWiFiDisplaySinkQueryDecoderAudioCodecs() API call. The application is
 * then able to decide which codecs to advertise support for.
 *
 * For more information on advertising codec support, see
 * VNCWiFiDisplaySinkAddAudioCodec().
 *
 * \param decoderFactoryContext The VNCWiFiDisplayDecoderFactoryContext that
 * was passed to VNCWiFiDisplayDecoderFactoryInitialize().
 * \param codec The audio codec to add. Ownership is not transferred to the
 * SDK.
 * \param codecSize The size of codec.
 *
 * \return VNCWiFiDisplayDecoderErrorNone The audio codec was added
 * successfully.
 * \return VNCWiFiDisplayDecoderErrorIllegalState The method was called from
 * outside the VNCWiFiDisplayDecoderFactoryInitialize() method.
 * \return VNCWiFiDisplayDecoderErrorInvalidParameter The audio codec could not
 * be added, as it has an invalid audio format or audio modes.
 * \return VNCWiFiDisplayDecoderErrorCodecAlreadyExists An audio codec with the
 * format given has already been added.
 * \return VNCWiFiDisplayDecoderErrorCodecMissingMandatoryMode The audio codec
 * could not be added, as it supports the VNCWiFiDisplayAudioFormatLPCM format
 * but not the mandatory VNCWiFiDisplayAudioModeLPCM_48000_16_2 mode.
 *
 * \see VNCWiFiDisplayAudioCodec, VNCWiFiDisplayAudioFormat
 * \see VNCWiFiDisplayDecoderAudioCodecSelectedCallback
 * \see AudioModeLPCM
 * \see AudioModeAAC
 * \see AudioModeAC3
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderAddAudioCodec(
    VNCWiFiDisplayDecoderFactoryContext decoderFactoryContext,
    const VNCWiFiDisplayAudioCodec *codec,
    size_t codecSize);

/**
 * \brief Indicates that this decoder supports a particular H.264 codec.
 *
 * This method should be called only from
 * VNCWiFiDisplayDecoderFactoryInitialize(). It is not possible to subsequently
 * alter the list of supported codecs.
 *
 * The list of supported codecs will be made available to the application using
 * the VNCWiFiDisplaySinkQueryDecoderH264Codecs() API call. The application is
 * then able to decide which codecs to advertise support for.
 *
 * For more information on advertising codec support, see
 * VNCWiFiDisplaySinkAddH264Codec().
 *
 * \param decoderFactoryContext The VNCWiFiDisplayDecoderFactoryContext that
 * was passed to VNCWiFiDisplayDecoderFactoryInitialize().
 * \param codec The H.264 codec to add. Ownership is not transferred to the
 * SDK.
 * \param codecSize The size of codec.
 *
 * \return VNCWiFiDisplayDecoderErrorNone The H.264 codec was added
 * successfully.
 * \return VNCWiFiDisplayDecoderErrorIllegalState The H.264 codec could not be
 * added, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayDecoderErrorInvalidParameter The H.264 codec could not
 * be added, as one or more details are invalid.
 * \return VNCWiFiDisplayDecoderErrorCodecAlreadyExists An H.264 codec has
 * already been added with the same details.
 *
 * \see VNCWiFiDisplayH264Profile, VNCWiFiDisplayH264Level
 * \see VNCWiFiDisplayH264Codec, VNCWiFiDisplayDecoderH264CodecSelectedCallback
 */
typedef VNCWiFiDisplayDecoderError VNCCALL
VNCWiFiDisplayDecoderAddH264Codec(
    VNCWiFiDisplayDecoderFactoryContext decoderFactoryContext,
    const VNCWiFiDisplayH264Codec *codec,
    size_t codecSize);

/**
 * \brief Called by the decoder to instruct the SDK to send a WiFi Display
 * IDR request.
 *
 * The source should respond with an IDR frame, allowing the decoder to
 * recover from dropped packets and corruption.
 *
 * If the WiFi Display session is not in the playing state, then this
 * method does nothing.
 *
 * \param decoderContext The VNCWiFiDisplayDecoderContext that was passed to
 * VNCWiFiDisplayDecoderCreate().
 */
typedef void VNCCALL
VNCWiFiDisplayDecoderRequestIDR(
    VNCWiFiDisplayDecoderContext decoderContext);

/**
 * \brief Called by the decoder to notify the SDK of a change in the H.264
 * stream resolution.
 *
 * This should be called when the first video frame is decoded, with the
 * dimensions of that frame. It should also be called if the resolution changes
 * during the course of a connection.
 *
 * Calling this API will cause the VNCWiFiDisplaySinkResolutionSelectedCallback
 * to be invoked, with the width and height provided here.
 *
 * This API may be called from any thread, and the result will be passed
 * asynchronously to the WiFi Display Sink.
 *
 * For performance reasons, avoid calling this API unnecessarily -- it is only
 * necessary to call this when the resolution changes, not for every frame
 * which is received.
 *
 * \param width The new X resolution of the H.264 stream.
 * \param height The new Y resolution of the H.264 stream.
 */
typedef void VNCCALL
VNCWiFiDisplayDecoderNotifyResolutionSelected(
    VNCWiFiDisplayDecoderContext decoderContext,
    vnc_uint16_t width,
    vnc_uint16_t height);

/**
 * \brief APIs provided by the SDK for calling by the decoder.
 */
typedef struct {
  /**
   * Called by the decoder factory to write to the SDK's log.
   */
  VNCWiFiDisplayDecoderFactoryLog *vncWiFiDisplayDecoderFactoryLog;
  /**
   * Called by the decoder to allocate memory whose ownership is to be
   * transferred to the SDK.
   */
  VNCWiFiDisplayDecoderFactoryAlloc *vncWiFiDisplayDecoderFactoryAlloc;
  /**
   * Called by the decoder to allocate memory whose ownership is to be
   * transferred to the SDK.
   */
  VNCWiFiDisplayDecoderFactoryAllocString *vncWiFiDisplayDecoderFactoryAllocString;
  /**
   * Called by the decoder to free memory whose ownership has not been
   * transferred to the SDK.
   */
  VNCWiFiDisplayDecoderFactoryFree *vncWiFiDisplayDecoderFactoryFree;
  /**
   * Called by the decoder to write to the SDK's log.
   */
  VNCWiFiDisplayDecoderLog *vncWiFiDisplayDecoderLog;
  /**
   * Called by the decoder to check whether a particular features is licensed.
   */
  VNCWiFiDisplayDecoderLocalFeatureCheck *vncWiFiDisplayDecoderLocalFeatureCheck;
  /**
   * Called by the decoder (from the RTP client thread) to return ownership of
   * an RTP packet to the SDK.
   */
  VNCWiFiDisplayDecoderRecycleRTPPacket *vncWiFiDisplayDecoderRecycleRTPPacket;
  /**
   * Called by the decoder (from any thread) to post a message to the RTP
   * client thread.
   */
  VNCWiFiDisplayDecoderPostRTPThreadMessage *vncWiFiDisplayDecoderPostRTPThreadMessage;
  /**
   * Called by the decoder (from VNCWiFiDisplayDecoderFactoryInitialize()) to
   * indicate that it supports a particular audio codec.
   */
  VNCWiFiDisplayDecoderAddAudioCodec *vncWiFiDisplayDecoderAddAudioCodec;
  /**
   * Called by the decoder (from VNCWiFiDisplayDecoderFactoryInitialize()) to
   * indicate that it supports a particular H.264 codec.
   */
  VNCWiFiDisplayDecoderAddH264Codec *vncWiFiDisplayDecoderAddH264Codec;
  /**
   * Called by the decoder to request an IDR frame.
   */
  VNCWiFiDisplayDecoderRequestIDR *vncWiFiDisplayDecoderRequestIDR;
  /**
   * Called by the decoder to notify the sink that the video resolution has
   * changed.
   */
  VNCWiFiDisplayDecoderNotifyResolutionSelected *vncWiFiDisplayDecoderNotifyResolutionSelected;

} VNCWiFiDisplayDecoderSupportingAPI;


/**
 * \brief The type of the entry point to be exported by a decoder DLL
 * or shared object.
 *
 * The exported symbol name of the entry point should be
 * VNCWiFiDisplayDecoderFactoryInitialize.
 *
 * \param decoderFactoryContext Opaque data type to be provided to
 * applicable supporting APIs.
 * \if ANDROID
 * \param decoderViewerContext The Android viewer context provided by the SDK,
 * as a VNCWiFiDisplayDecoderAndroidViewerContext instance. Your decoder
 * implementation can use this context to communicate with your application.
 * See \ref section_android_viewer_context for more information.
 * \else
 * \param decoderViewerContext Opaque data type as provided by the
 * viewer application to the SDK when the decoder was registered or
 * loaded.
 * \endif
 * \param pDecoderInterface Structure to contain the addresses of the
 * decoders' implementation functions.  The implementation of
 * VNCWiFiDisplayDecoderFactoryInitialize() must fill in this structure before
 * returning.
 * \param decoderInterfaceSize The size of *pDecoderInterface.
 * \param pDecoderSupportingAPI Structure containing functions provided by the
 * SDK for calling by the decoder.
 * \param decoderSupportingAPISize The size of *pDecoderSupportingAPI.
 *
 * \return The new decoder object.  The SDK will call
 * VNCWiFiDisplayDecoderFactoryTerminate() when this object is no longer
 * required. Returning NULL here indicates that an error has occurred, and the
 * decoder factory will not be initialised.
 *
 * \if ANDROID
 * \see VNCWiFiDisplayDecoderAndroidViewerContext
 * \endif
 */
typedef VNCWiFiDisplayDecoderFactory *VNCCALL
VNCWiFiDisplayDecoderFactoryInitializeType(
    VNCWiFiDisplayDecoderFactoryContext decoderFactoryContext,
    VNCWiFiDisplayDecoderViewerContext decoderViewerContext,
    VNCWiFiDisplayDecoderInterface *pDecoderInterface,
    size_t decoderInterfaceSize,
    const VNCWiFiDisplayDecoderSupportingAPI *pDecoderSupportingAPI,
    size_t decoderSupportingAPISize);

#ifdef __cplusplus
}
#endif

#endif // VNCWIFIDISPLAYDECODER_H_70571462_0029_4B31_B79F_6E6701933255

